/** @file   lighttable.cpp
 * @brief   Implementation of LightTable class.
 * @version $Revision: 1.5 $
 * @date    $Date: 2006/06/21 16:09:39 $
 * @author  Tomi Lamminsaari
 */


#include "lighttable.h" // class's header file
#include "GfxManager.h"
#include "settings.h"
#include "LightSourceModel.h"
#include "www_assert.h"
using std::vector;
using eng2d::Vec2D;


namespace WeWantWar {


LightSource::LightSource( const Vec2D& aPosition, int aModelUid, int aIntensity ) :
  iPosition( aPosition ),
  iModelUid( aModelUid ),
  iIntensity( aIntensity )
{
}

LightSource::~LightSource()
{
}

int LightSource::modelUid() const
{
  return iModelUid;
}

Vec2D LightSource::position() const
{
  return iPosition;
}

int LightSource::intensity() const
{
  return iIntensity;
}


///
/// Constants and members
/// =====================

///
/// Methods
/// =======

/** Constructor.
 */
LightTable::LightTable()
{
}

LightTable::~LightTable()
{
  this->cleanup();
}

/** Removes all the lightsources
 */
void LightTable::cleanup()
{
  for ( int i=0; i < iLightsources.size(); i++ ) {
    if ( iLightsources.at(i) != 0 ) {
      delete iLightsources.at(i);
      iLightsources.at(i) = 0;
    }
  }
  iLightsources.clear();
  
  std::map<int, LightSourceModel*>::const_iterator it = iLightModels.begin();
  while ( it != iLightModels.end() ) {
    delete it->second;
    it++;
  }
  iLightModels.clear();
}



/** Adds new lightsource
 */
void LightTable::add( LightSource* aLightSource )
{
  if ( aLightSource != 0 ) {
    iLightsources.push_back( aLightSource );
  }
}



/** Registers new lightsource model.
 */
void LightTable::registerLightSourceModel( LightSourceModel* aModel )
{
  if ( aModel != 0 ) {
    iLightModels[ aModel->modelUid() ] = aModel;
  }
}


/** Draws all the lightsources
 */
void LightTable::redraw( BITMAP* aCanvas, const Vec2D& aOffset )
{
  
  if ( Settings::lightSources == false || aCanvas == 0 ) {
    return;
  }
  for ( int i=0; i < iLightsources.size(); i++ ) {
    LightSource* lightSource = iLightsources.at(i);
    
    // Find the lighting model the current lightsource.
    std::map<int,LightSourceModel*>::const_iterator it =
            iLightModels.find( lightSource->modelUid() );
    if ( it != iLightModels.end() ) {
      // Apply the light model.
      LightSourceModel* model = it->second;
      Vec2D pos = lightSource->position();
      pos -= aOffset;
      model->redraw( pos, aCanvas, lightSource->intensity() );
    }
  }
}

/** Removes the light source.
 */
void LightTable::removeLightSource( int aIndex )
{
  if ( aIndex < 0 || aIndex >= iLightsources.size() ) {
    return;
  }
  vector<LightSource*>::iterator it = iLightsources.begin() + aIndex;
  delete (*it);
  iLightsources.erase( it );
}

/** Removes the light source model.
 */
void LightTable::removeLightSourceModel( int aUid )
{
  std::map<int, LightSourceModel*>::iterator it = iLightModels.find( aUid );
  if ( it != iLightModels.end() ) {
    delete it->second;
    iLightModels.erase( it );
  }
}

/** Returns the LightSources
 */
LightSource* LightTable::getLightSource( int aIndex ) const
{
  if ( aIndex < 0 || aIndex >= iLightsources.size() ) {
    return 0;
  }
  return iLightsources.at( aIndex );
}

/** Returns the light models.
 */
LightSourceModel* LightTable::findLightSourceModel( int aUid ) const
{
  std::map<int, LightSourceModel*>::const_iterator it = iLightModels.find(aUid);
  if ( it != iLightModels.end() ) {
    return it->second;
  } else {
    return 0;
  }
}

} // end of namespace
